/*
 * This File is one of the examples from Java Object Oriented Programming
 * Do not reproduce this code for others or use it in a commercial setting without prior permission from the author.
 */

package salesApplication;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import productsFinal.Eraser;
import productsFinal.Notepad;
import productsFinal.Pen;
import productsFinal.Pencil;
import productsFinal.PencilCase;
import productsFinal.WritingPack;

/**
 *
 * @author Matthew Gregory
 * @website www.matthewgregory-author.com.au
 *
 */
public class FileDataSource implements DataSource {
    //The advantage of this data source is that it will be saved to file
    //so it will be available even after the JVM has shut down.
    
    
    //Local copy of all the price lists.
    Map<String,PriceList> lists=new HashMap<>();
    
    public FileDataSource(){
        //If we already have the file...load the data.
        if (Files.exists(Paths.get("mydatasource"))){
            loadList();
        } else {
        //If not create a demo list to start, but don't save it until requested
            lists.put("Demo",createDemoPriceList());
        }
    }
    
    public void saveList(String name, PriceList priceList){
        //Put this price list into our data source.
        lists.put(name, priceList);
        //Save our datasource
        try (FileOutputStream fos = new FileOutputStream("mydatasource");
            //The file is saved in this projects folder by default.
            //In my case this is: C:\...\Java Object Oriented Programming\Projects\Chapter8 - Developing Interfaces
            ObjectOutputStream out = new ObjectOutputStream(fos);){
            out.writeObject(lists);
        } catch (IOException ex) {
            System.out.println("Unable to find destination file.");
            System.out.println(ex);
        }
    }
    
    public PriceList retrieveList(String name){
        //Only loading the data if we don't already have the price list they are looking for
        if (!lists.containsKey(name)) {
            loadList();
        }
        //Return the price list they are looking for.
        return lists.get(name);
    }
    
    public void loadList(){
        try (FileInputStream fis = new FileInputStream("mydatasource")){
            ObjectInputStream in = new ObjectInputStream(fis);
            lists = (Map<String,PriceList>) in.readObject();
        } catch (ClassNotFoundException | IOException ex) {
            System.out.println("Unable to find source file.");
            System.out.println(ex);
        } 
    }
    
    public void getListNames(){
        System.out.println("Current Price List Available: "+lists.keySet());
    }
    
    private static PriceList createDemoPriceList() {
        //Create an instance of pen
        Pen myPen = new Pen("Super Pen");
        myPen.type = "Felt-tip";
        myPen.enterDescription("This new pen is just amazing you need to try it!");
        myPen.setPrice(5.00);

        //Create an instance of Pencil
        Pencil myPencil = new Pencil("Custom Pencil");
        myPencil.enterDescription("This pencil is designed to stay sharp longer, and is comfortable in the hand.");
        myPencil.setPrice(2.00);

        //Create an instance of Notepad
        Notepad myNotepad = new Notepad("Student Special");
        myNotepad.enterDescription("240 leaf A4 lecture pad.");
        myNotepad.setPrice(3.50);

        //Create an instance of Eraser
        Eraser myEraser = new Eraser("New and Improved Eraser");
        myEraser.enterDescription("A long lasting smudgeless eraser for all kinds of pencils.");
        myEraser.setPrice(1.50);

        //Create an instance of PencilCase
        PencilCase myPencilCase = new PencilCase("Basic Pencil Case");
        myPencilCase.enterDescription("A small single zip plastic pencil case with lettering for names.");
        myPencilCase.setPrice(3.00);

        //Create and instance of WritingPack including 2x myPens, myEraser, myPencil and myPencilCase
        WritingPack myWritingPack = new WritingPack("Special Writing Pack");
        myWritingPack.pen1 = myPen;
        myWritingPack.pen2 = myPen;
        myWritingPack.pencil = myPencil;
        myWritingPack.eraser = myEraser;
        myWritingPack.pencilCase = myPencilCase;
        myWritingPack.enterDescription("WritingPack including two super pens, a custom pencil,"
                + " a new and improved eraser and a basic pencil case.");
        myWritingPack.setPrice(15.00);

        //create a price list with all these products in it and print it out
        PriceList demoPriceList = new PriceList();
        demoPriceList.addProduct(myPen);
        demoPriceList.addProduct(myPencil);
        demoPriceList.addProduct(myEraser);
        demoPriceList.addProduct(myPencilCase);
        demoPriceList.addProduct(myNotepad);
        demoPriceList.addProduct(myWritingPack);

        return demoPriceList;
    }
}
